methods {
    function premiumFunction(bytes32, uint8, uint256, uint8, uint256[2], uint256[2][2], uint256[2], uint256[4], uint256[2]) external;
    function batchPremiumFunction(bytes32[], uint8[], uint256[], uint8[], uint256[2][], uint256[2][2][], uint256[2][], uint256[4][], uint256[2][]) external;
    function epochSubmit(bytes32, uint256, uint64, uint256[2], uint256[2][2], uint256[2], uint256[4], uint256[2]) external;
    function _computeLatticeHash(bytes32, uint256, uint8, uint8) internal returns bytes32;
    function _computeReward(uint256) internal returns (uint256, uint256);
    function withdrawETH() external;
    function withdrawCHG() external;
    function registerStealthAddress(bytes32) external;
}

rule no_reentrancy(method f) {
    env e; calldataarg args; storage initial = lastStorage;
    f(e, args);
    assert lastStorage == initial, "Reentrancy detected";
}

rule compute_lattice_hash_deterministic(bytes32 input, uint256 tick, uint8 depth, uint8 instanceId) {
    env e;
    bytes32 result1 = _computeLatticeHash(e, input, tick, depth, instanceId);
    bytes32 result2 = _computeLatticeHash(e, input, tick, depth, instanceId);
    assert result1 == result2, "Non-deterministic hash";
}

rule zchgsnark_proof_validity(bytes32 latticeHash, uint256 computeUnitsTotal, uint64 epoch, uint256[2] a, uint256[2][2] b, uint256[2] c, uint256[4] zkpInput, uint256[2] commitment) {
    env e;
    epochSubmit(e, latticeHash, computeUnitsTotal, epoch, a, b, c, zkpInput, commitment);
    assert zkVerifier.verifyProof(a, b, c, zkpInput, commitment), "Invalid zCHGsnark proof";
    assert bytes32(zkpInput[0]) == latticeHash, "Proof hash mismatch";
}

rule bulletproof_commitment_valid(uint256 computeUnitsTotal, uint256[2] commitment) {
    env e;
    // Simplified: Assume verifier checks commitment
    assert computeUnitsTotal < 1e18, "Invalid compute units";
    assert zkVerifier.verifyProof(_, _, _, _, commitment), "Invalid Pedersen commitment";
}

rule stealth_address_privacy(address user, bytes32 stealthAddress) {
    env e;
    registerStealthAddress(e, stealthAddress);
    assert stealthAddresses[user] == stealthAddress, "Stealth address not registered";
    assert stealthAddress != bytes32(0), "Invalid stealth address";
}

rule superposition_constraint(bytes32 input, uint256 tick, uint8 instanceId) {
    env e;
    bytes32 hash = _computeLatticeHash(e, input, tick, 1, instanceId);
    assert hash[2] == hash[6], "Superposition resonance violated (D_3, D_7)";
}

rule epoch_consistency(uint64 epoch, bytes32 latticeHash, uint256 computeUnitsTotal, uint256[2] a, uint256[2][2] b, uint256[2] c, uint256[4] zkpInput, uint256[2] commitment) {
    env e;
    uint64 oldEpoch = currentEpoch;
    bytes32 oldSnapshot = lastSnapshot;
    epochSubmit(e, latticeHash, computeUnitsTotal, epoch, a, b, c, zkpInput, commitment);
    assert currentEpoch == oldEpoch + 1, "Epoch not incremented";
    assert epochSnapshots[epoch] == latticeHash, "Snapshot not stored";
    assert lastSnapshot == latticeHash, "Last snapshot not updated";
}

rule reward_fairness(uint256 computeUnitsUsed) {
    env e;
    uint256 ethReward; uint256 chgReward;
    (ethReward, chgReward) = _computeReward(e, computeUnitsUsed);
    assert computeUnitsUsed < 1e18, "Compute units overflow";
    assert ethReward == (1e15 * 1e6) / (computeUnitsUsed + 1), "Invalid ETH reward";
    assert chgReward == (1000 * 1e6) / (computeUnitsUsed + 1), "Invalid CHG reward";
}

rule withdraw_eth_safe() {
    env e;
    uint256 balanceBefore = balancesETH[e.msg.sender];
    bytes32 stealthAddress = stealthAddresses[e.msg.sender];
    uint256 ethBefore = e.msg.sender.balance;
    withdrawETH(e);
    assert balanceBefore > 0 => e.msg.sender.balance > ethBefore, "ETH withdrawal failed";
    assert balancesETH[e.msg.sender] == 0, "ETH balance not cleared";
    assert stealthAddress != bytes32(0), "Stealth address required";
}

rule withdraw_chg_safe() {
    env e;
    uint256 balanceBefore = balancesCHG[e.msg.sender];
    bytes32 stealthAddress = stealthAddresses[e.msg.sender];
    uint256 chgBefore = chgToken.balanceOf(e.msg.sender);
    withdrawCHG(e);
    assert balanceBefore > 0 => chgToken.balanceOf(e.msg.sender) > chgBefore, "CHG withdrawal failed";
    assert balancesCHG[e.msg.sender] == 0, "CHG balance not cleared";
    assert stealthAddress != bytes32(0), "Stealth address required";
}